/*
	Input Capture Module with automatic upscale and downscale and co-op multitasking...
		
	capture.c
	
	by Mauro Grassi, September 2010.
	
*/

#include "HardwareProfile.h"
#include "capture.h"
#include <math.h>
#include "main.h"
#include "mystring.h"
#include "clock.h"
#include "uart.h"

#pragma udata capturedata

volatile CAPTURE_OBJECT	captureObjectNumber[NUM_CAPTURE_OBJECTS];
volatile unsigned char  captureObjectInputNumber[TOTAL_PHYSICAL_CHANNELS];		/* holds the mapping from physical channels back to logical channels, ie. capture objects */
volatile unsigned char  totalCaptureObjects;
volatile unsigned char  timer3OverFlow;
volatile unsigned char  timer3Refs;
volatile unsigned char  timer5OverFlow;
volatile unsigned char  timer5Refs;
unsigned char           timer5Uses;
unsigned char			timer3Uses;

#pragma udata

#if(IS_PC_HOST)

#else

#pragma code usercode

const rom unsigned char captureChannels[CAPTURE_LOGICAL_CHANNELS]=
{
	/* maps logical channel numbers, eg, D0-D5 to physical Capture channels */
	4, 5, 6, 7, 2, 3
};
#endif

unsigned char capture_Task(CAPTURE_OBJECT* captureObject)
{
	/* 
	
		returns 1 if a new frequency has been read, otherwise 0 
		
	*/
		
	float 			f;
	float			overflow;
	unsigned char 	result;
	unsigned char	cIndex;
	
	
	#if 0
	
	static volatile unsigned char  ostate;

	if(ostate!=captureObject->captureState)
	{
		putrsUART("State: ");
		uftoa(tempString, (float)captureObject->captureState, 3);
		putsUART(tempString);
		putrsUART(" ");
		
		putrsUART("InputNumber: ");
		uftoa(tempString, (float)captureObject->inputNumber, 3);
		putsUART(tempString);
		putrsUART(" ");
		
		putrsUART(" CCPCON: ");
		uftoa(tempString, (float)captureObject->ccpcon, 3);
		putsUART(tempString);
		putrsUART(" ");
		
		putrsUART(" Scaled Timer Frequency: ");
		uftoa(tempString, (float)captureObject->scaledTimerFrequency/1000.0, 3);
		putsUART(tempString);
		putrsUART(" ");
		
		putrsUART(" CT0: ");
		uftoa(tempString, (float)captureObject->captureT[0], 3);
		putsUART(tempString);
		putrsUART(" ");
		
		putrsUART(" CT1: ");
		uftoa(tempString, (float)captureObject->captureT[1], 3);
		putsUART(tempString);
		putrsUART("\r\n");
		
		ostate=captureObject->captureState;	
	}
	#endif

	result=0;
	if((captureObject->mode & 3)==NO_FREQUENCY_MODE)cIndex=1; else cIndex=0;
	
	switch(captureObject->captureState)
	{
			case INITIAL_CAPTURE_STATE:
			default:			
					
				calculateTimerUses();
				if((timer3Uses<=0)&&(T3CON!=0))
				{
					T3CON=0;
				}
				else
				if((timer3Uses>0)&&(T3CON==0))
				{
					T3CON=0x03;
				}
				
				if((timer5Uses<=0)&&(T5CON!=0))
				{
					T5CON=0;
				}
				else
				if((timer5Uses>0)&&(T5CON==0))
				{
					T5CON=0x03;
				}

				switch(captureObject->inputNumber)
				{
					default:
					case 0:
						/* OFF */
						break;
					
					case 1:
						#if 0
						PIE1bits.CCP1IE=0;
						captureObject->captureIndex=cIndex;
						CCP1CON=captureObject->ccpcon;
						IPR1bits.CCP1IP=INTERRUPT_PRIORITY_CAPTURE;
						PIR1bits.CCP1IF=0;
						PIE1bits.CCP1IE=1;
						#endif
						break;
			
					case 2:
						PIE2bits.CCP2IE=0;
						captureObject->captureIndex=cIndex;
						CCP2CON=captureObject->ccpcon;
						IPR2bits.CCP2IP=INTERRUPT_PRIORITY_CAPTURE;
						PIR2bits.CCP2IF=0;
						PIE2bits.CCP2IE=1;
						break;
			
					case 3:
						PIE4bits.CCP3IE=0;
						captureObject->captureIndex=cIndex;
						CCP3CON=captureObject->ccpcon;
						IPR4bits.CCP3IP=INTERRUPT_PRIORITY_CAPTURE;
						PIR4bits.CCP3IF=0;
						PIE4bits.CCP3IE=1;
						break;
					
					case 4:
						PIE4bits.CCP4IE=0;
						captureObject->captureIndex=cIndex;
						CCP4CON=captureObject->ccpcon;
						IPR4bits.CCP4IP=INTERRUPT_PRIORITY_CAPTURE;
						PIR4bits.CCP4IF=0;
						PIE4bits.CCP4IE=1;
						break;
							
					case 5:
						PIE4bits.CCP5IE=0;
						captureObject->captureIndex=cIndex;
						CCP5CON=captureObject->ccpcon;
						IPR4bits.CCP5IP=INTERRUPT_PRIORITY_CAPTURE;
						PIR4bits.CCP5IF=0;
						PIE4bits.CCP5IE=1;
						break;
			
					case 6:
						PIE4bits.CCP6IE=0;
						captureObject->captureIndex=cIndex;
						CCP6CON=captureObject->ccpcon;
						IPR4bits.CCP6IP=INTERRUPT_PRIORITY_CAPTURE;
						PIR4bits.CCP6IF=0;
						PIE4bits.CCP6IE=1;
						break;
			
					case 7:
						PIE4bits.CCP7IE=0;
						captureObject->captureIndex=cIndex;
						CCP7CON=captureObject->ccpcon;
						IPR4bits.CCP7IP=INTERRUPT_PRIORITY_CAPTURE;
						PIR4bits.CCP7IF=0;
						PIE4bits.CCP7IE=1;
						break;
					}
					captureObject->captureState=WAIT_CAPTURE_STATE;
					break;
					
			case WAIT_CAPTURE_STATE:
				switch(captureObject->inputNumber)
				{
					default:
					case 0:
						/* OFF */
						captureObject->captureState=FINISH_CAPTURE_STATE;
						break;
						
					case 1:
						if(PIE1bits.CCP1IE==1)
						{
						
						}
						else
						{
							captureObject->captureState=FINISH_CAPTURE_STATE;
						}
						break;
			
					case 2:
						if(PIE2bits.CCP2IE==1)
						{
						
						}
						else
						{
							captureObject->captureState=FINISH_CAPTURE_STATE;
						}
						break;
			
					case 3:
						if(PIE4bits.CCP3IE==1)
						{
						
						}
						else
						{
							captureObject->captureState=FINISH_CAPTURE_STATE;
						}
						break;
					
					case 4:
						if(PIE4bits.CCP4IE==1)
						{
						
						}
						else
						{
							captureObject->captureState=FINISH_CAPTURE_STATE;
						}
						break;
							
					case 5:
						if(PIE4bits.CCP5IE==1)
						{
						
						}
						else
						{
							captureObject->captureState=FINISH_CAPTURE_STATE;
						}
						break;
			
					case 6:
						if(PIE4bits.CCP6IE==1)
						{
						
						}
						else
						{
							captureObject->captureState=FINISH_CAPTURE_STATE;
						}
						break;
			
					case 7:
						if(PIE4bits.CCP7IE==1)
						{
						
						}
						else
						{
							captureObject->captureState=FINISH_CAPTURE_STATE;
						}
						break;
					}
					break;
			
			case FINISH_CAPTURE_STATE:
					switch(captureObject->inputNumber)
					{
						case 0:
						default:
							break;
							
						case 1:
							CCP1CON=0x00;
							break;
							
						case 2:
							CCP2CON=0x00;
							break;
							
						case 3:
							CCP3CON=0x00;
							break;
							
						case 4:
							CCP4CON=0x00;
							break;
							
						case 5:
							CCP5CON=0x00;
							break;
							
						case 6:
							CCP6CON=0x00;
							break;
							
						case 7:
							CCP7CON=0x00;
							break;
					}
					
					switch(captureObject->mode & 3)
					{
						default:
						case NO_FREQUENCY_MODE:
						
							break;
							
						case LOW_FREQUENCY_MODE:						
							overflow=CAPTURE_OVERFLOW_LIMIT_LOW;
							if(captureObject->captureTimer[0]>captureObject->captureTimer[1])
							{						
								f=(float)(captureObject->captureT[0]+65536.0*captureObject->captureTimer[0]);
								f-=(float)(captureObject->captureT[1]+65536.0*captureObject->captureTimer[1]);
							}
							else
							if(captureObject->captureTimer[1]>captureObject->captureTimer[0])
							{
								f=(float)(captureObject->captureT[1]+65536.0*captureObject->captureTimer[1]);
								f-=(float)(captureObject->captureT[0]+65536.0*captureObject->captureTimer[0]);							
							}
							else
							{
								f=(float)captureObject->captureT[0]-(float)captureObject->captureT[1];
							}
							break;
						
						case MEDIUM_FREQUENCY_MODE:							
						case HIGH_FREQUENCY_MODE:
							if((captureObject->mode & 3)==HIGH_FREQUENCY_MODE)overflow=CAPTURE_OVERFLOW_LIMIT_HIGH;
							else overflow=CAPTURE_OVERFLOW_LIMIT_MEDIUM;
							if(captureObject->captureT[0]>captureObject->captureT[1])
							{
								f=(float)(captureObject->captureT[0]-captureObject->captureT[1]);
							}
							else
							{
								f=(float)((unsigned long)captureObject->captureT[0]+0x10000-(unsigned long)captureObject->captureT[1]);
							}
							break;	
						}
					
					/* Get the Absolute Value */
					if(f<0.0)f=-f;
					
					switch(captureObject->mode & 3)
					{
						default:
							break;
						
						case NO_FREQUENCY_MODE:
							break;
						
						case LOW_FREQUENCY_MODE:
						case MEDIUM_FREQUENCY_MODE:
						case HIGH_FREQUENCY_MODE:
					
								/* Check for Overflow or Underflow */
								if((captureObject->captureTimer[1]!=captureObject->captureTimer[0])&&((captureObject->mode & 3)!=LOW_FREQUENCY_MODE))
								{
									/* Underflow detected, the input frequency is too low for this mode! */
									f=0.0;
									if(captureObject->mode & ALLOW_FREQUENCY_SCALING)
									{
										downScaleCaptureObject(captureObject);
									}
								}
								else
								if((f>=0.0)&&(f<=overflow))
								{
									/* Overflow detected, the input frequency is too high for this mode ! */
									f=0.0;
									if(captureObject->mode & ALLOW_FREQUENCY_SCALING)
									{
										upScaleCaptureObject(captureObject);
									}					
								}									
								if(f>0.0)
								{
									f=(captureObject->scaledTimerFrequency/(f));
									captureObject->frequencyOrCounter=f;
								}
								break;				
					}
					result=1;
					captureObject->captureState=IDLE_CAPTURE_STATE;
					break;
					
				case IDLE_CAPTURE_STATE:
					break;
	}
	return result;
}

void doCaptureInterrupt(void)
{
	unsigned char   captureObjectNum;

	if((PIE5bits.TMR5IE)&&(PIR5bits.TMR5IF))
	{
		timer5OverFlow++;
		if(timer5Refs>0)timer5Refs--;
		else PIE5bits.TMR5IE=0;		
		PIR5bits.TMR5IF=0;	
	}

	if((PIE2bits.TMR3IE)&&(PIR2bits.TMR3IF))
	{
		timer3OverFlow++;
		if(timer3Refs>0)timer3Refs--;
		else PIE2bits.TMR3IE=0;
		PIR2bits.TMR3IF=0;	
	}
	
	#if 0
	if((PIE1bits.CCP1IE)&&(PIR1bits.CCP1IF))
	{
		captureObjectNum=captureObjectInputNumber[0];
		if((captureObjectNum>=1)&&(captureObjectNum<=NUM_CAPTURE_OBJECTS))
		{		
		captureObjectNum--;
		if(captureObjectNumber[captureObjectNum].captureIndex & 1)
		{
			captureObjectNumber[captureObjectNum].captureT[0]=(unsigned int)CCPR1L;
			captureObjectNumber[captureObjectNum].captureT[0]+=(unsigned int)(((unsigned long)CCPR1H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].frequencyOrCounter+=1.0;
						break;
						
				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer3OverFlow;
						break;						
			}
			PIE1bits.CCP1IE=0;
		}
		else
		{		
			captureObjectNumber[captureObjectNum].captureT[1]=(unsigned int)CCPR1L;
			captureObjectNumber[captureObjectNum].captureT[1]+=(unsigned int)(((unsigned long)CCPR1H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;
						
				case NO_FREQUENCY_MODE:
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer3OverFlow;
						timer3Refs++;
						break;						
			}
		}
		captureObjectNumber[captureObjectNum].captureIndex^=1;
		}
		else
		{
			PIE1bits.CCP1IE=0;
		}
		PIR1bits.CCP1IF=0;
	}
	
	#endif
	
	if((PIE2bits.CCP2IE)&&(PIR2bits.CCP2IF))
	{	
		captureObjectNum=captureObjectInputNumber[1];
		if((captureObjectNum>=1)&&(captureObjectNum<=NUM_CAPTURE_OBJECTS))
		{		
		captureObjectNum--;
		if(captureObjectNumber[captureObjectNum].captureIndex & 1)
		{
			captureObjectNumber[captureObjectNum].captureT[0]=(unsigned int)CCPR2L;
			captureObjectNumber[captureObjectNum].captureT[0]+=(unsigned int)(((unsigned long)CCPR2H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].frequencyOrCounter+=1.0;
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer3OverFlow;
						break;						
			}
			PIE2bits.CCP2IE=0;
		}
		else
		{		
			captureObjectNumber[captureObjectNum].captureT[1]=(unsigned int)CCPR2L;
			captureObjectNumber[captureObjectNum].captureT[1]+=(unsigned int)(((unsigned long)CCPR2H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer3OverFlow;
						timer3Refs++;
						break;						
			}
		}
		captureObjectNumber[captureObjectNum].captureIndex^=1;
		}
		else
		{
			PIE2bits.CCP2IE=0;
		}		
		PIR2bits.CCP2IF=0;
	}

	if((PIE4bits.CCP3IE)&&(PIR4bits.CCP3IF))
	{
		captureObjectNum=captureObjectInputNumber[2];
		if((captureObjectNum>=1)&&(captureObjectNum<=NUM_CAPTURE_OBJECTS))
		{		
		captureObjectNum--;
		if(captureObjectNumber[captureObjectNum].captureIndex & 1)
		{
			captureObjectNumber[captureObjectNum].captureT[0]=(unsigned int)CCPR3L;
			captureObjectNumber[captureObjectNum].captureT[0]+=(unsigned int)(((unsigned long)CCPR3H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].frequencyOrCounter+=1.0;
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer3OverFlow;
						break;						
			}
			PIE4bits.CCP3IE=0;
		}
		else
		{		
			captureObjectNumber[captureObjectNum].captureT[1]=(unsigned int)CCPR3L;
			captureObjectNumber[captureObjectNum].captureT[1]+=(unsigned int)(((unsigned long)CCPR3H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer3OverFlow;
						timer3Refs++;
						break;						
			}
		}
		captureObjectNumber[captureObjectNum].captureIndex^=1;
		}
		else
		{
			PIE4bits.CCP3IE=0;
		}
		PIR4bits.CCP3IF=0;
	}

	if((PIE4bits.CCP4IE)&&(PIR4bits.CCP4IF))
	{
		captureObjectNum=captureObjectInputNumber[3];
		if((captureObjectNum>=1)&&(captureObjectNum<=NUM_CAPTURE_OBJECTS))
		{		
		captureObjectNum--;
		if(captureObjectNumber[captureObjectNum].captureIndex & 1)
		{
			captureObjectNumber[captureObjectNum].captureT[0]=(unsigned int)CCPR4L;
			captureObjectNumber[captureObjectNum].captureT[0]+=(unsigned int)(((unsigned int)CCPR4H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].frequencyOrCounter+=1.0;
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer3OverFlow;
						break;						
			}
			PIE4bits.CCP4IE=0;
		}
		else
		{		
			captureObjectNumber[captureObjectNum].captureT[1]=(unsigned int)CCPR4L;
			captureObjectNumber[captureObjectNum].captureT[1]+=(unsigned int)(((unsigned int)CCPR4H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer3OverFlow;
						timer3Refs++;
						break;						
			}
		}
		captureObjectNumber[captureObjectNum].captureIndex^=1;
		}
		else
		{
			PIE4bits.CCP4IE=0;
		}
		PIR4bits.CCP4IF=0;
	}

	if((PIE4bits.CCP5IE)&&(PIR4bits.CCP5IF))
	{
		captureObjectNum=captureObjectInputNumber[4];
		if((captureObjectNum>=1)&&(captureObjectNum<=NUM_CAPTURE_OBJECTS))
		{		
		captureObjectNum--;
		if(captureObjectNumber[captureObjectNum].captureIndex & 1)
		{
			captureObjectNumber[captureObjectNum].captureT[0]=(unsigned int)CCPR5L;
			captureObjectNumber[captureObjectNum].captureT[0]+=(unsigned int)(((unsigned long)CCPR5H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].frequencyOrCounter+=1.0;
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer5OverFlow;
						break;						
			}
			PIE4bits.CCP5IE=0;
		}
		else
		{		
			captureObjectNumber[captureObjectNum].captureT[1]=(unsigned int)CCPR5L;
			captureObjectNumber[captureObjectNum].captureT[1]+=(unsigned int)(((unsigned long)CCPR5H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer5OverFlow;
						timer5Refs++;
						break;						
			}
		}
		captureObjectNumber[captureObjectNum].captureIndex^=1;
		}
		else
		{
			PIE4bits.CCP5IE=0;
		}
		PIR4bits.CCP5IF=0;
	}
	
	if((PIE4bits.CCP6IE)&&(PIR4bits.CCP6IF))
	{
		captureObjectNum=captureObjectInputNumber[5];
		if((captureObjectNum>=1)&&(captureObjectNum<=NUM_CAPTURE_OBJECTS))
		{		
		captureObjectNum--;
		if(captureObjectNumber[captureObjectNum].captureIndex & 1)
		{
			captureObjectNumber[captureObjectNum].captureT[0]=(unsigned int)CCPR6L;
			captureObjectNumber[captureObjectNum].captureT[0]+=(unsigned int)(((unsigned long)CCPR6H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].frequencyOrCounter+=1.0;
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer5OverFlow;
						break;						
			}
			PIE4bits.CCP6IE=0;
		}
		else
		{		
			captureObjectNumber[captureObjectNum].captureT[1]=(unsigned int)CCPR6L;
			captureObjectNumber[captureObjectNum].captureT[1]+=(unsigned int)(((unsigned long)CCPR6H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer5OverFlow;
						timer5Refs++;
						break;						
			}
		}
		captureObjectNumber[captureObjectNum].captureIndex^=1;
		}
		else
		{
			PIE4bits.CCP6IE=0;
		}
		PIR4bits.CCP6IF=0;
	}
	
	if((PIE4bits.CCP7IE)&&(PIR4bits.CCP7IF))
	{
		captureObjectNum=captureObjectInputNumber[6];
		if((captureObjectNum>=1)&&(captureObjectNum<=NUM_CAPTURE_OBJECTS))
		{		
		captureObjectNum--;
		if(captureObjectNumber[captureObjectNum].captureIndex & 1)
		{
			captureObjectNumber[captureObjectNum].captureT[0]=(unsigned int)CCPR7L;
			captureObjectNumber[captureObjectNum].captureT[0]+=(unsigned int)(((unsigned long)CCPR7H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].frequencyOrCounter+=1.0;
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[0]=timer5OverFlow;
						break;						
			}
			PIE4bits.CCP7IE=0;
		}
		else
		{		
			captureObjectNumber[captureObjectNum].captureT[1]=(unsigned int)CCPR7L;
			captureObjectNumber[captureObjectNum].captureT[1]+=(unsigned int)(((unsigned long)CCPR7H)<<8);
			switch(captureObjectNumber[captureObjectNum].mode & 3)
			{
				default:
						break;

				case NO_FREQUENCY_MODE:
						break;

				case LOW_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer1OverFlow;
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
						captureObjectNumber[captureObjectNum].captureTimer[1]=timer5OverFlow;
						timer5Refs++;
						break;						
			}
		}
		captureObjectNumber[captureObjectNum].captureIndex^=1;
		}
		else
		{
			PIE4bits.CCP7IE=0;
		}
		PIR4bits.CCP7IF=0;
	}

	if(timer3Refs>0)
	{
		PIR2bits.TMR3IF=0;
		PIE2bits.TMR3IE=1;
	}
	
	if(timer5Refs>0)
	{
		PIR5bits.TMR5IF=0;
		PIE5bits.TMR5IE=1;
	}
}

void capture_FullTask(void)
{
	unsigned int i;
	
	i=0;
	while(i<totalCaptureObjects)
	{
		captureObjectNumber[i].status=capture_Task((CAPTURE_OBJECT*)&captureObjectNumber[i]);
		i++;
	}
}

void powerCapture(unsigned char on)
{
	/* Power Up Or Down the System */
	if(on)
	{
		#if(USE_PMDIS)
			PMDIS2bits.TMR5MD=0;
			PMDIS1bits.TMR3MD=0;
			PMDIS0bits.ECCP2MD=0;
			PMDIS0bits.ECCP3MD=0;
			PMDIS3bits.CCP4MD=0;
			PMDIS3bits.CCP5MD=0;
			PMDIS3bits.CCP6MD=0;
			PMDIS3bits.CCP7MD=0;
		#endif		
	}
	else
	{
		#if(USE_PMDIS)
			PMDIS2bits.TMR5MD=1;
			PMDIS1bits.TMR3MD=1;
			PMDIS0bits.ECCP2MD=1;
			PMDIS0bits.ECCP3MD=1;
			PMDIS3bits.CCP4MD=1;
			PMDIS3bits.CCP5MD=1;
			PMDIS3bits.CCP6MD=1;
			PMDIS3bits.CCP7MD=1;		
		#endif	
	}
}

void clearCapture(void)
{
	/* Initialize System */	
	int i;
	for(i=0; i<TOTAL_PHYSICAL_CHANNELS; i++)
	{
		captureObjectInputNumber[i]=0;
	}
	
	for(i=0; i<NUM_CAPTURE_OBJECTS; i++)
	{
		mem_set((void*)&captureObjectNumber[i], 0, sizeof(CAPTURE_OBJECT));
		captureObjectNumber[i].scaledTimerFrequency=1.0;
		captureObjectNumber[i].mode=NO_FREQUENCY_MODE;
		captureObjectNumber[i].captureState=INITIAL_CAPTURE_STATE;
	}
	totalCaptureObjects=0;
}

void initCapture(void)
{
	powerCapture(1);
	timer3OverFlow=0;
	timer3Refs=0;
	timer5OverFlow=0;
	timer5Refs=0;
	timer5Uses=0;
	timer3Uses=0;
	IPR2bits.TMR3IP=INTERRUPT_PRIORITY_CAPTURE;
	IPR5bits.TMR5IP=INTERRUPT_PRIORITY_CAPTURE;
	PIR2bits.TMR3IF=0;
	PIR5bits.TMR5IF=0;
	PIE5bits.TMR5IE=0;
	PIE2bits.TMR3IE=0;
	T3CON=0;
	T5CON=0;
}

void calculateTimerUses(void)
{
	int 			i;
	int 			num;

	timer3Uses=0;
	timer5Uses=0;		
	for(i=0; i<TOTAL_PHYSICAL_CHANNELS; i++)
	{
		if(captureObjectInputNumber[i]!=0)
		{
			num=captureObjectInputNumber[i]-1;
			switch(captureObjectNumber[num].mode & 3)
			{
				case NO_FREQUENCY_MODE:
				case LOW_FREQUENCY_MODE:
						break;
						
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:	
						switch(captureObjectNumber[num].inputNumber)
						{
							default:
								break;
								
							case 1:
								break;
								
							case 2:
							case 3:
							case 4:
								timer3Uses++;
								break;
							
							case 5:
							case 6:
							case 7:
								timer5Uses++;
								break;
			
						}
			}
		}
	}		
}

void reopenCloseCaptureObject(unsigned char on)
{
	int 			i;
	int 			num;
		
	for(i=0; i<TOTAL_PHYSICAL_CHANNELS; i++)
	{
		if(captureObjectInputNumber[i]!=0)
		{
			num=captureObjectInputNumber[i]-1;
			if(on)
			{
				openCaptureObject(i+1, captureObjectNumber[num].mode);
			}
			else
			{
				internalCloseCaptureObject(i+1);
			}
		}
	}
}

float readCaptureObjectChannel(unsigned char physicalChannel, unsigned char* ready)
{
	float f;
	unsigned char num;
	
	f=0.0;
	*ready=0;
	
	if((physicalChannel>=2)&&(physicalChannel<=TOTAL_PHYSICAL_CHANNELS))
	{			
		num=captureObjectInputNumber[physicalChannel-1]-1;	
		if((num>=0)&&(num<NUM_CAPTURE_OBJECTS))
		{	
			/* Read the value */
			f=captureObjectNumber[num].frequencyOrCounter;
			/* Now retrigger it */		
			switch(captureObjectNumber[num].mode & 3)
			{
				default:
					break;
				
				case NO_FREQUENCY_MODE:
				case LOW_FREQUENCY_MODE:
				case MEDIUM_FREQUENCY_MODE:
				case HIGH_FREQUENCY_MODE:
					if(captureObjectNumber[num].captureState==IDLE_CAPTURE_STATE)
					{
						captureObjectNumber[num].captureState=INITIAL_CAPTURE_STATE;
						*ready=1;
					}
					break;
			}
		}
	}
	return f;
}

#if 0
void showCaptureObjects(void)
{
	int i;
	putrsUART((const   rom char*)"Total Capture Objects: ");
	disWUART(totalCaptureObjects);
	putrsUART((const   rom char*)"\r\n");
	for(i=0; i<totalCaptureObjects; i++)
	{	
		disWUART(i);
		putrsUART((const   rom char*)" : CCPCON: ");
		disWUART(captureObjectNumber[i].ccpcon);
		putrsUART((const   rom char*)" ScaledFreq: ");
		uftoa(tempString, (float)captureObjectNumber[i].scaledTimerFrequency/1000.0, 3);
		putsUART(tempString);
		putrsUART((const   rom char*)" kHz Mode: ");
		disWUART(captureObjectNumber[i].mode);
		putrsUART((const   rom char*)" InN: ");
		disWUART(captureObjectNumber[i].inputNumber);
		putrsUART((const   rom char*)" CT0: ");
		disWUART(captureObjectNumber[i].captureT[0]);
		putrsUART((const   rom char*)" CT1: ");
		disWUART(captureObjectNumber[i].captureT[1]);
		putrsUART((const   rom char*)"\r\n");		
	}
}

void showAllCaptureObjects(void)
{
	/* show the state of the capture system... */
	int i;
	
	putrsUART((const   rom char*)"Capture Object Input Number Table: \r\n");
	putrsUART((const rom char*)"TMR3OverFlow: ");
	disWUART(timer3OverFlow);
	putrsUART((const rom char*)" TMR3Refs: ");
	disWUART(timer3Refs);
	putrsUART((const rom char*)" TMR3IE: ");
	disWUART(PIE2bits.TMR3IE);
	putrsUART((const rom char*)" TMR3 Uses: ");
	disWUART(timer3Uses);
	putrsUART((const rom char*)"\r\n");
	
	putrsUART((const rom char*)" TMR5OverFlow: ");
	disWUART(timer5OverFlow);
	putrsUART((const rom char*)" TMR5Refs: ");
	disWUART(timer5Refs);
	putrsUART((const rom char*)" TMR5IE: ");
	disWUART(PIE5bits.TMR5IE);
	putrsUART((const rom char*)" TMR5 Uses: ");
	disWUART(timer5Uses);
	putrsUART((const rom char*)"\r\n");
	
	for(i=0; i<TOTAL_PHYSICAL_CHANNELS; i++)
	{
		disWUART((unsigned int)i);
		putrsUART((const   rom char*)" : ");
		disWUART((unsigned int)captureObjectInputNumber[i]);
		putrsUART((const   rom char*)"\r\n");
	}
	showCaptureObjects();
}
#endif

int upScaleCaptureObject(CAPTURE_OBJECT* captureObject)
{
	int i;
	
	i=1;
	switch(captureObject->mode & 3)
	{
		default:
			/* do nothing! */
			break;
			
		case LOW_FREQUENCY_MODE:
			i=captureObject->mode & 0xFC;
			i=openCaptureObject(captureObjectNumber->inputNumber, i+MEDIUM_FREQUENCY_MODE);
			break;
			
		case MEDIUM_FREQUENCY_MODE:
			i=captureObject->mode & 0xFC;
			i=openCaptureObject(captureObjectNumber->inputNumber, i+HIGH_FREQUENCY_MODE);
			break;
	}
	return i;
}

int downScaleCaptureObject(CAPTURE_OBJECT* captureObject)
{
	int i;
	
	i=1;
	switch(captureObject->mode & 3)
	{
		default:
			/* do nothing! */
			break;
						
		case MEDIUM_FREQUENCY_MODE:
			i=captureObject->mode & 0xFC;
			i=openCaptureObject(captureObjectNumber->inputNumber, i+LOW_FREQUENCY_MODE);
			break;

		case HIGH_FREQUENCY_MODE:
			i=captureObject->mode & 0xFC;
			i=openCaptureObject(captureObjectNumber->inputNumber, i+MEDIUM_FREQUENCY_MODE);
			break;
	}
	return i;
}

int openCaptureObject(unsigned char physicalChannel, unsigned char mode)
{	
	/* 
		Configure a Capture Object to read frequency in mode mode, connected to physical input number physicalChannel...

		returns:

			 1 if ok 
			 0 otherwise (there may not be enough logical channels for this...)
			
		mode can be:
		
			HIGH_FREQUENCY_MODE can be used for high frequency inputs
			MEDIUM_FREQUENCY_MODE for med frequencies...  
			LOW_FREQUENCY_MODE can be used for low frequency inputs  
			
			NO_FREQUENCY_MODE: used for counting events...
			
	    it can also have the 
	       
	       ALLOW_FREQUENCY_SCALING set...
	       
		physicalChannel:
		
			can be one of 10 physical inputs, numbered from 1 to 10
			
	*/
	
	int 			i, j, z;
	int 			result;
	unsigned char 	ccpt;
			
	/* 
		
		First Check if the Physical Channel is already in use...
				
	*/
	
	result=0;
	z=0;
	if((physicalChannel>=2)&&(physicalChannel<=TOTAL_PHYSICAL_CHANNELS))
	{	
		if(captureObjectInputNumber[physicalChannel-1]==0)
		{
			/* 	
				The channel is not currently used, so try to use it...
			*/	
			if(totalCaptureObjects<(NUM_CAPTURE_OBJECTS-1))
			{
				j=totalCaptureObjects;		
				totalCaptureObjects++;
				/* zero new ones, not others (for up and down scaling) */
				z=1;
			}
		}
		else
		{
			j=(captureObjectInputNumber[physicalChannel-1]-1);
		}
		
		/* 
			
			The channel to update is j if (j!=-1), so update the settings for it...
				
		*/
	
		if((j>=0)&&(j<NUM_CAPTURE_OBJECTS))
		{
			
			result=1;
			
			if(z)captureObjectNumber[j].frequencyOrCounter=0.0;
			
			switch(mode & 3)
			{
				default:
				case NO_FREQUENCY_MODE:
					captureObjectNumber[j].frequencyOrCounter=0.0;
					if(mode & COUNT_RISING_EDGES)
						captureObjectNumber[j].ccpcon=0x05;
					else 
						captureObjectNumber[j].ccpcon=0x04;
						
					captureObjectNumber[j].scaledTimerFrequency=1.0;
					break;
	
				case LOW_FREQUENCY_MODE:
					switch(physicalChannel)
					{
						default:
							break;
	
						case 1:
							#if 0
							ccpt=CCPTMRS0;
							ccpt&=0xF8;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=32768.0;
							captureObjectNumber[j].ccpcon=0x05;
							/* Assumes that Timer1 is already active */
							#endif
							break;
							
						case 2:
							ccpt=CCPTMRS0;
							ccpt&=0xC7;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=32768.0;	
							captureObjectNumber[j].ccpcon=0x05;
							/* Assumes that Timer1 is already active */
							break;
						
						case 3:
							ccpt=CCPTMRS0;
							ccpt&=0x3F;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=32768.0;
							captureObjectNumber[j].ccpcon=0x05;
							/* Assumes that Timer1 is already active */
							break;
							
						case 4:
							ccpt=CCPTMRS1;
							ccpt&=0xFC;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=32768.0;
							captureObjectNumber[j].ccpcon=0x05;		
							/* Assumes that Timer1 is already active */
							break;
							
						case 5:
							ccpt=CCPTMRS1;
							ccpt&=0xFB;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=32768.0;
							captureObjectNumber[j].ccpcon=0x05;
							/* Assumes that Timer1 is already active */
							break;
												
						case 6:
							ccpt=CCPTMRS1;
							ccpt&=0xEF;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=32768.0;
							captureObjectNumber[j].ccpcon=0x05;
							/* Assumes that Timer1 is already active */
							break;
							
						case 7:
							ccpt=CCPTMRS1;
							ccpt&=0x3F;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=32768.0;
							captureObjectNumber[j].ccpcon=0x05;
							/* Assumes that Timer1 is already active */
							break;		
					}
					break;
					
				case MEDIUM_FREQUENCY_MODE:
					switch(physicalChannel)
					{
						default:
							break;
	
						case 1:
							#if 0
							ccpt=CCPTMRS0;
							ccpt&=0xF8;
							ccpt|=0x01;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock());
							captureObjectNumber[j].ccpcon=0x05;		
							#endif
							break;
							
						case 2:
							ccpt=CCPTMRS0;
							ccpt&=0xC7;
							ccpt|=0x08;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock());	
							captureObjectNumber[j].ccpcon=0x05;
							break;
						
						case 3:
							ccpt=CCPTMRS0;
							ccpt&=0x3F;
							ccpt|=0x40;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock());
							captureObjectNumber[j].ccpcon=0x05;
							break;
							
						case 4:
							ccpt=CCPTMRS1;
							ccpt&=0xFC;
							ccpt|=0x01;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock());
							captureObjectNumber[j].ccpcon=0x05;
							break;
							
						case 5:
							ccpt=CCPTMRS1;
							ccpt&=0xFB;
							ccpt|=0x04;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock());
							captureObjectNumber[j].ccpcon=0x05;
							if(z)timer5Uses++;
							break;
												
						case 6:
							ccpt=CCPTMRS1;
							ccpt&=0xEF;
							ccpt|=0x10;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock());
							captureObjectNumber[j].ccpcon=0x05;
							if(z)timer5Uses++;
							break;
							
						case 7:
							ccpt=CCPTMRS1;
							ccpt&=0x3F;
							ccpt|=0x40;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock());
							captureObjectNumber[j].ccpcon=0x05;
							if(z)timer5Uses++;
							break;
					}
					break;		
					
				case HIGH_FREQUENCY_MODE:
					switch(physicalChannel)
					{
						default:
							break;
	
						case 1:
							#if 0
							ccpt=CCPTMRS0;
							ccpt&=0xF8;
							ccpt|=0x01;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock()*16.0);
							captureObjectNumber[j].ccpcon=0x07;
							#endif
							break;
							
						case 2:
							ccpt=CCPTMRS0;
							ccpt&=0xC7;
							ccpt|=0x08;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock()*16.0);	
							captureObjectNumber[j].ccpcon=0x07;
							break;
						
						case 3:
							ccpt=CCPTMRS0;
							ccpt&=0x3F;
							ccpt|=0x40;
							CCPTMRS0=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock()*16.0);
							captureObjectNumber[j].ccpcon=0x07;
							break;
							
						case 4:
							ccpt=CCPTMRS1;
							ccpt&=0xFC;
							ccpt|=0x01;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock()*16.0);
							captureObjectNumber[j].ccpcon=0x07;
							break;
							
						case 5:
							ccpt=CCPTMRS1;
							ccpt&=0xFB;
							ccpt|=0x04;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock()*16.0);
							captureObjectNumber[j].ccpcon=0x07;
							if(z)timer5Uses++;
							break;
												
						case 6:
							ccpt=CCPTMRS1;
							ccpt&=0xEF;
							ccpt|=0x10;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock()*16.0);
							captureObjectNumber[j].ccpcon=0x07;
							if(z)timer5Uses++;
							break;
							
						case 7:
							ccpt=CCPTMRS1;
							ccpt&=0x3F;
							ccpt|=0x40;
							CCPTMRS1=ccpt;
							captureObjectNumber[j].scaledTimerFrequency=(GetInstructionClock()*16.0);
							captureObjectNumber[j].ccpcon=0x07;
							if(z)timer5Uses++;
							break;
					}
					break;
			}
			
			/* Preapare the hardware pin */
			switch(physicalChannel)
					{
						default:
							break;
	
						case 1:
							break;
							
						case 2:
							#if(USE_PMDIS)
								PMDIS0bits.ECCP2MD=0;
							#endif
							RPINR8=0;
							TRISAbits.TRISA0=1;
							break;
						
						case 3:
							#if(USE_PMDIS)
								PMDIS0bits.ECCP3MD=0;
							#endif
							RPINR9=1;
							TRISAbits.TRISA1=1;
							break;
							
						case 4:
							#if(USE_PMDIS)
								PMDIS3bits.CCP4MD=0;
							#endif
							TRISBbits.TRISB4=1;
							break;
							
						case 5:
							#if(USE_PMDIS)
								PMDIS3bits.CCP5MD=0;
							#endif
							TRISBbits.TRISB5=1;
							break;
												
						case 6:
							#if(USE_PMDIS)
								PMDIS3bits.CCP6MD=0;
							#endif
							TRISBbits.TRISB6=1;
							break;
							
						case 7:
							#if(USE_PMDIS)
								PMDIS3bits.CCP7MD=0;
							#endif
							TRISBbits.TRISB7=1;
							break;		
			}
			captureObjectNumber[j].mode=mode;
			captureObjectNumber[j].inputNumber=physicalChannel;
			captureObjectNumber[j].captureT[0]=0;
			captureObjectNumber[j].captureT[1]=0;
			captureObjectNumber[j].captureIndex=0;
			captureObjectNumber[j].captureState=INITIAL_CAPTURE_STATE;
			captureObjectInputNumber[physicalChannel-1]=j+1;
		
			calculateTimerUses();	
			if(timer3Uses>0)
			{
				T3CON=0x03;			
			}			
			
			if(timer5Uses>0)
			{
				T5CON=0x03;
			}
		}
		else
		{
					
		}			
	}
	return result;
}

int deleteCaptureObject(unsigned char num)
{
	int i;
	
	if((num>=0)&&(num<(totalCaptureObjects)))
	{
		if((num>=0)&&(num<(totalCaptureObjects-1)))
		{
			i=(num+1);
			while(i<(totalCaptureObjects))
			{
				captureObjectNumber[i-1].mode=captureObjectNumber[i].mode;
				captureObjectNumber[i-1].inputNumber=captureObjectNumber[i].inputNumber;
				captureObjectNumber[i-1].captureT[0]=captureObjectNumber[i].captureT[0];
				captureObjectNumber[i-1].captureT[1]=captureObjectNumber[i].captureT[1];
				captureObjectNumber[i-1].captureIndex=captureObjectNumber[i].captureIndex;
				captureObjectNumber[i-1].captureState=captureObjectNumber[i].captureState;
				captureObjectNumber[i-1].scaledTimerFrequency=captureObjectNumber[i].scaledTimerFrequency;
				captureObjectNumber[i-1].ccpcon=captureObjectNumber[i].ccpcon;
				captureObjectNumber[i-1].status=captureObjectNumber[i].status;
				captureObjectInputNumber[captureObjectNumber[i].inputNumber-1]--;
				i++;
			}
		}
	totalCaptureObjects--;
	}
}

int internalCloseCaptureObject(unsigned char physicalChannel)
{
			switch(physicalChannel)
			{
				default:
				case 1:
					break;
							
				case 2:
					RPINR8=0xFF;
					TRISAbits.TRISA0=1;
					CCP2CON=0;
					#if(USE_PMDIS)
						PMDIS0bits.ECCP2MD=1;
					#endif
					break;
					
				case 3:
					RPINR9=0xFF;
					TRISAbits.TRISA1=1;
					CCP3CON=0;
					#if(USE_PMDIS)
						PMDIS0bits.ECCP3MD=1;
					#endif
					break;
							
				case 4:
					TRISBbits.TRISB4=1;
					CCP4CON=0;
					#if(USE_PMDIS)
						PMDIS3bits.CCP4MD=1;
					#endif
					break;
							
				case 5:
					TRISBbits.TRISB5=1;
					CCP5CON=0;
					#if(USE_PMDIS)
						PMDIS3bits.CCP5MD=1;
					#endif
					break;
												
				case 6:
					TRISBbits.TRISB6=1;
					CCP6CON=0;
					#if(USE_PMDIS)
						PMDIS3bits.CCP6MD=1;
					#endif
					break;
							
				case 7:
					TRISBbits.TRISB7=1;
					CCP7CON=0;
					#if(USE_PMDIS)
						PMDIS3bits.CCP7MD=1;
					#endif
					break;
	}
}

int closeCaptureObject(unsigned char physicalChannel)
{
	unsigned char mode;
	
	if((physicalChannel>=2)&&(physicalChannel<=TOTAL_PHYSICAL_CHANNELS))
	{	
		if(captureObjectInputNumber[physicalChannel-1]>0)
		{
			mode=captureObjectNumber[captureObjectInputNumber[physicalChannel-1]-1].mode;
			deleteCaptureObject(captureObjectInputNumber[physicalChannel-1]-1);
			captureObjectInputNumber[physicalChannel-1]=0;
			/* Do the actual hardware closing... */
			internalCloseCaptureObject(physicalChannel);				
		}
	}		
}
